<#
    .SYNOPSIS
    Disounts VHD images mounted with Mount-TriageVhdx.ps1

    .DESCRIPTION
    When Mount-TriageVhdx.ps1 was used to mount a collection of KAPE images to directories in the specified Mount Directory, this script will unmount them and remove the created mount point directories.
    
    Mounting and dismounting require administrator permissions, so the PowerShell terminal must run as administrator.
    .PARAMETER VhdDirectory
    Specifies the directory where the unmounted .vhd or .vhdx files are stored. Files should be in the root of this directory. Script does not recurse into subfolders.

    .PARAMETER MountDirectory
    Specifies the directory where mount points for each mounted image will be created. Within the mount directory, a new directory (mount point) was created for each image file. This script will delete them.
            
    .PARAMETER StripDateTime
    An optional switch that specifies whether to remove dates in the format YYYY-MM-DDTHHMMSS from the beginning of the disk image file name when determining the associated computer name. The default is false. If the switch is specified, the script will remove the date and an underscore from the beginning of the file name before then calculating the computer name.

    Script assumes file names follow this format: SourceComputerName-triage.vhdx
    
    .EXAMPLE
    PS> Dismount-TriageVhdx.ps1 -VhdDirectory "D:\VHDXTriageImagesFolder\" -MountDirectory "D:\MountedImagesFolder\"
    This example dismounts disk images located in D:\VHDXTriageImagesFolder\ and removes the associated mount points in D:\MountedImagesFolder\.

    .EXAMPLE
    PS> Dismount-TriageVhdx.ps1 -VhdDirectory "D:\VHDXTriageImagesFolder\" -MountDirectory "D:\MountedImagesFolder\" -StripDateTime
    This example dismounts disk images located in D:\VHDXTriageImagesFolder\ and removes the associated mount points in D:\MountedImagesFolder\ removing the date and time stamp from the beginning of the file name before calculating the computer name.        
    
    .NOTES
    Written by Dr. Sorot Panichprecha and Steve Anson. This program is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version. This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details. The GNU General Public License can be found at <https://www.gnu.org/licenses/>.
#>
param (
    [Parameter(Position=0,mandatory=$true)]
    $VhdDirectory,
    [Parameter(Position=1,mandatory=$true)]
    $MountDirectory,
    [Parameter(mandatory=$false)]
    [switch]$StripDateTime
)

# Check if the current PowerShell session is running as administrator
try {
    $currentPrincipal = [Security.Principal.WindowsPrincipal] [Security.Principal.WindowsIdentity]::GetCurrent()
    if (-NOT $currentPrincipal.IsInRole([Security.Principal.WindowsBuiltInRole] "Administrator")) {
        throw 
    }
}
catch {
    Write-Host -ForegroundColor Red "ERROR: This script must be run as Administrator."
    exit 1
}

# Check if the VHD directory exists. If not, exit the program.
if (-Not (Test-Path -Path $VhdDirectory)) {
    Write-Host "ERROR: $VhdDirectory does not exist"
    Exit 1
}

# Check if the mount directory exists. If not, exit the program.
if (-Not (Test-Path -Path $MountDirectory)) {
    Write-Host "ERROR: $MountDirectory does not exist"
    Exit 1
}

# Dismount each VHD file and delete associated mount point directories
Get-ChildItem -Path $VhdDirectory -Include *.vhd,*.vhdx -Recurse | ForEach-Object {
    Write-Host "Dismounting $_"
    try {
        [void](Dismount-DiskImage -ImagePath $_)
    }
    catch {
        Write-Host "Error occurred while dismounting disk image: $_"
    }
    #Remove mount point directory
    $VhdFileName = $_.Name
    if ($StripDateTime) {
        # Extract the first 17 characters of the file name
        $dateTimeString = $VhdFileName.Substring(0, 17)
        $format = "yyyy-MM-ddTHHmmss"
        $date = New-Object DateTime
        # Check if the first 17 characters of the file name are a properly formatted date and time stamp
        if (-not [DateTime]::TryParseExact($dateTimeString, $format, [Globalization.CultureInfo]::InvariantCulture, [Globalization.DateTimeStyles]::None, [ref]$date)) {
            Write-Host -ForegroundColor Yellow "WARNING: Unexpected file name format for $VhdFileName. The file name does not begin with a properly formatted date and time stamp."
        }
        # Cut the first 18 characters from the file name for further processing to find the appropriate computer name
        $VhdFileName = ($VhdFileName).substring(18)
    }
    if ($VhdFileName -like "*-triage.vhd") {
        $ComputerName = ($VhdFileName).substring(0, $VhdFileName.Length - 11)
    }
    elseif ($VhdFileName -like "*-triage.vhdx") {
        $ComputerName = ($VhdFileName).substring(0, $VhdFileName.Length - 12)
    }
    elseif ($VhdFileName -like "*.vhd") {
        $ComputerName = ($VhdFileName).substring(0, $VhdFileName.Length - 4)
    }
    elseif ($VhdFileName -like "*.vhdx") {
        $ComputerName = ($VhdFileName).substring(0, $VhdFileName.Length - 5)
    }
    else {
        Write-Host -ForegroundColor Yellow "WARNING: Unexpected file name format for $VhdFileName."
        Write-Host -ForegroundColor Yellow "This file will be ignored."
        continue
    }
    if (Test-Path -Path "$MountDirectory/$ComputerName") {
        If (-Not (Test-Path -Path "$MountDirectory/$ComputerName\*")) {
        Write-Host "Deleting empty folder $MountDirectory$ComputerName"
        Remove-Item -Path "$MountDirectory/$ComputerName"
        }
        Else {
            Write-host -f Yellow "$MountDirectory$ComputerName is not empty. This folder will not be deleted."
             }
    }
}